package org.seasar.webwork.nazo;

import com.opensymphony.xwork.config.Configuration;
import com.opensymphony.xwork.config.ConfigurationManager;
import com.opensymphony.xwork.config.RuntimeConfiguration;
import com.opensymphony.xwork.config.entities.ActionConfig;
import com.opensymphony.xwork.config.entities.PackageConfig;
import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import junit.framework.TestCase;
import org.seasar.extension.mock.servlet.MockServletContext;
import org.seasar.framework.container.S2Container;
import org.seasar.framework.container.impl.S2ContainerImpl;

public class RuntimeConfigurationImplTest extends TestCase {
    private S2Container container;

    public void setUp() {
	container = new S2ContainerImpl();
    }

    public void testGetActionConfigFromDelegate() {
	final ActionConfig expectedActionConfig = new ActionConfig();

	RuntimeConfiguration runtimeConfiguration = new RuntimeConfiguration() {
		public ActionConfig getActionConfig(String namespace, String name) {
		    assertEquals("testNamespace", namespace);
		    assertEquals("testName", name);
		    return expectedActionConfig;
		}

		public Map getActionConfigs() {
		    throw new UnsupportedOperationException();
		}
	    };
	RuntimeConfiguration nazoRuntimeConfiguration = new RuntimeConfigurationImpl(runtimeConfiguration, container);
	ActionConfig actualActionConfig = nazoRuntimeConfiguration.getActionConfig("testNamespace", "testName");
	assertEquals(expectedActionConfig, actualActionConfig);
    }

    public void testNazoPackageNotFound() {
	RuntimeConfiguration runtimeConfiguration = new RuntimeConfiguration() {
		public ActionConfig getActionConfig(String namespace, String name) {
		    return null;
		}

		public Map getActionConfigs() {
		    throw new UnsupportedOperationException();
		}
	    };
	RuntimeConfiguration nazoRuntimeConfiguration = new RuntimeConfigurationImpl(runtimeConfiguration, container);
	ActionConfig actionConfig = nazoRuntimeConfiguration.getActionConfig("testNamespace", "testName");
	assertNull(actionConfig);
    }

    public void testNamespaceMismatch() {
	Configuration configuration = ConfigurationManager.getConfiguration();
	PackageConfig nazoPackageConfig = new PackageConfig("nazo", "testNamespace", true, null);
	configuration.addPackageConfig(nazoPackageConfig.getName(), nazoPackageConfig);

	RuntimeConfiguration runtimeConfiguration = new RuntimeConfiguration() {
		public ActionConfig getActionConfig(String namespace, String name) {
		    return null;
		}

		public Map getActionConfigs() {
		    throw new UnsupportedOperationException();
		}
	    };
	RuntimeConfiguration nazoRuntimeConfiguration = new RuntimeConfigurationImpl(runtimeConfiguration, container);
	ActionConfig actionConfig = nazoRuntimeConfiguration.getActionConfig("unknownNamespace", "testName");
	assertNull(actionConfig);
    }

    public void testNazoPackageIsNotAbstract() {
	Configuration configuration = ConfigurationManager.getConfiguration();
	PackageConfig nazoPackageConfig = new PackageConfig("nazo", "testNamespace", false, null);
	configuration.addPackageConfig(nazoPackageConfig.getName(), nazoPackageConfig);

	RuntimeConfiguration runtimeConfiguration = new RuntimeConfiguration() {
		public ActionConfig getActionConfig(String namespace, String name) {
		    return null;
		}

		public Map getActionConfigs() {
		    throw new UnsupportedOperationException();
		}
	    };
	RuntimeConfiguration nazoRuntimeConfiguration = new RuntimeConfigurationImpl(runtimeConfiguration, container);
	ActionConfig actionConfig = nazoRuntimeConfiguration.getActionConfig("testNamespace", "testName");
	assertNull(actionConfig);
    }


    public void testGroovyScriptNotFound() {
	container.setServletContext(new MockServletContext("/s2-nazo-webwork") {
		public InputStream getResourceAsStream(String name) {
		    assertEquals("/WEB-INF/nazo/testName.groovy", name);
		    return null;
		}
	    });

	Configuration configuration = ConfigurationManager.getConfiguration();
	PackageConfig nazoPackageConfig = new PackageConfig("nazo", "testNamespace", true, null);
	configuration.addPackageConfig(nazoPackageConfig.getName(), nazoPackageConfig);

	RuntimeConfiguration runtimeConfiguration = new RuntimeConfiguration() {
		public ActionConfig getActionConfig(String namespace, String name) {
		    return null;
		}

		public Map getActionConfigs() {
		    throw new UnsupportedOperationException();
		}
	    };
	RuntimeConfiguration nazoRuntimeConfiguration = new RuntimeConfigurationImpl(runtimeConfiguration, container);
	ActionConfig actionConfig = nazoRuntimeConfiguration.getActionConfig("testNamespace", "testName");
	assertNull(actionConfig);
    }

    public void testGetGroovyActionConfig() {
	container.setServletContext(new MockServletContext("/s2-nazo-webwork") {
		public InputStream getResourceAsStream(String name) {
		    assertEquals("/WEB-INF/nazo/testName.groovy", name);
		    return new ByteArrayInputStream(new byte[] {});
		}
	    });

	Configuration configuration = ConfigurationManager.getConfiguration();
	PackageConfig defaultPackageConfig = configuration.getPackageConfig("webwork-default");
	List parents = Collections.singletonList(defaultPackageConfig);
	PackageConfig nazoPackageConfig = new PackageConfig("nazo", "testNamespace", true, null, parents);
	nazoPackageConfig.setDefaultInterceptorRef("completeStack");
	configuration.addPackageConfig(nazoPackageConfig.getName(), nazoPackageConfig);

	RuntimeConfiguration runtimeConfiguration = new RuntimeConfiguration() {
		public ActionConfig getActionConfig(String namespace, String name) {
		    return null;
		}

		public Map getActionConfigs() {
		    throw new UnsupportedOperationException();
		}
	    };
	RuntimeConfiguration nazoRuntimeConfiguration = new RuntimeConfigurationImpl(runtimeConfiguration, container);
	ActionConfig actionConfig = nazoRuntimeConfiguration.getActionConfig("testNamespace", "testName");
	assertNotNull(actionConfig);
	assertEquals("/WEB-INF/nazo/testName.groovy", actionConfig.getClassName());
	assertNull(actionConfig.getExternalRefs());
	assertEquals(11, actionConfig.getInterceptors().size());
	assertNull(actionConfig.getMethodName());
	assertEquals("nazo", actionConfig.getPackageName());
	assertTrue(actionConfig.getParams().isEmpty());
	assertTrue(actionConfig.getResults() instanceof ResultConfigMap);
    }
}
